$ErrorActionPreference = "Stop"
Set-Location $PSScriptRoot

$log = Join-Path $PSScriptRoot "stem_mixer_install.log"
"=== AI Stem Mixer (PowerShell Launcher) ===" | Out-File -Encoding utf8 $log
"Started: $(Get-Date)" | Out-File -Append -Encoding utf8 $log
"Folder: $PSScriptRoot" | Out-File -Append -Encoding utf8 $log

function Log($msg) { $msg | Tee-Object -FilePath $log -Append | Out-Host }
function Fail($msg) {
  Log ""
  Log "ERROR: $msg"
  Log "See log: $log"
  try { Start-Process notepad.exe $log | Out-Null } catch {}
  Read-Host "Press Enter to close"
  exit 1
}

try {
  Log ""
  Log "Detecting Python (py / python / python3)..."
  $pyCmd = $null
  foreach ($c in @("py","python","python3")) {
    if (Get-Command $c -ErrorAction SilentlyContinue) { $pyCmd = $c; break }
  }
  if (-not $pyCmd) { Fail "No Python command found. Install Python 3 and check 'Add to PATH'." }
  Log "Using: $pyCmd"

  if (-not (Test-Path ".venv")) {
    Log "Creating venv (.venv)..."
    & $pyCmd -m venv .venv 2>&1 | Tee-Object -FilePath $log -Append | Out-Host
  } else {
    Log "Venv already exists."
  }

  $venvPy = Join-Path $PSScriptRoot ".venv\Scripts\python.exe"
  if (-not (Test-Path $venvPy)) { Fail "venv python not found: $venvPy" }

  Log "Installing dependencies (first run can take a while; Torch is large)..."
  & $venvPy -m pip install --upgrade pip 2>&1 | Tee-Object -FilePath $log -Append | Out-Host
  & $venvPy -m pip install -r requirements.txt 2>&1 | Tee-Object -FilePath $log -Append | Out-Host


Log "Ensuring torchcodec is NOT installed (it can break audio decode on some Windows/Python 3.13 setups)..."
try { & $venvPy -m pip uninstall -y torchcodec 2>&1 | Tee-Object -FilePath $log -Append | Out-Host } catch {}

  # Remove old port file
  $portFile = Join-Path $PSScriptRoot "server_port.txt"
  if (Test-Path $portFile) { Remove-Item $portFile -Force -ErrorAction SilentlyContinue }

  
# Force torchaudio backend away from torchcodec (even if present elsewhere)
$env:TORCHAUDIO_BACKEND = "soundfile"
$env:TORCHAUDIO_USE_TORCHCODEC = "0"

Log "Starting server..."
  # Start server in a child PowerShell so we can wait+open browser reliably
  $serverProc = Start-Process -FilePath $venvPy -ArgumentList "app.py" -WorkingDirectory $PSScriptRoot -PassThru -NoNewWindow `
    -RedirectStandardOutput (Join-Path $PSScriptRoot "server_stdout.log") -RedirectStandardError (Join-Path $PSScriptRoot "server_stderr.log")

  # Wait for port file and server to accept connections
  $port = $null
  $deadline = (Get-Date).AddSeconds(25)
  while ((Get-Date) -lt $deadline) {
    if ($serverProc.HasExited) { break }
    if ((-not $port) -and (Test-Path $portFile)) {
      try { $port = (Get-Content $portFile -Raw).Trim() } catch {}
    }
    if ($port) {
      try {
        $client = New-Object System.Net.Sockets.TcpClient
        $client.Connect("127.0.0.1", [int]$port)
        $client.Close()
        break
      } catch {}
    }
    Start-Sleep -Milliseconds 250
  }

  if ($serverProc.HasExited) {
    $err = ""
    try { $err = (Get-Content (Join-Path $PSScriptRoot "server_stderr.log") -Raw) } catch {}
    Fail "Server exited immediately. Details in server_stderr.log`n$err"
  }

  if (-not $port) { $port = "8080" } # fallback
  Log "Opening http://localhost:$port"
  Start-Process "http://localhost:$port/" | Out-Null

  Log "Server running. Close this window to stop."
  Log "(Logs: server_stdout.log / server_stderr.log)"
  Read-Host "Press Enter to stop the server"
  try { $serverProc.CloseMainWindow() | Out-Null } catch {}
  try { Stop-Process -Id $serverProc.Id -Force -ErrorAction SilentlyContinue } catch {}
  exit 0

} catch {
  Fail $_.Exception.Message
}